/**
 * Objeto que contiene métodos (funciones) para realizar operaciones diversas
 * aplicable en todos los ámbitos de la aplicación.
 * <br/>
 * La idea es que esté disponible para toda la aplicación llamando este objeto
 * cuando sea necesario.<br> <br>
 * Incluir por ejemplo: los cálculos de framingham (brasil y colombia), cálculo
 * de riesgo de infarto (especialidad de cardiología), etc.... pues estas
 * funcionalidades pueden ser usadas en otras parte del sitio.
 * 
 * <b>Evitar dependencia con alguna librería</b>, al menos que fuese necesario
 * 
 * TODO: separar funcionalidades por algún criterio, crear más objetos bajo
 * este criterio si es necesario
 * 
 * @type objecto JS
 */    
HERRAMIENTAS = {

    /**
     * Permite redondear números según <code>decimales</code> usando
     * redondeo clásico: >=0.5 aumenta unidad
     * 
     * @param {int} numero
     * @param {int} decimales
     * @returns {Number}
     */
    redondear: function( numero, decimales ) {

        if( isNaN( parseFloat( numero ) ) || isNaN( parseInt( decimales ) ) )
            throw new TypeError( "Uno de los parámetros no es válido" );

        return Math.round( numero * Math.pow( 10, decimales ) ) / Math.pow( 10, decimales );
    },

    /**
     * Calcula el IMC corporal, fórmula clásica: peso kgs/(altura mts)2
     * 
     * @param {int} peso peso en kilogramos
     * @param {int} estatura estatura <b>en metros<b>
     * @return {float} IMC con 1 decimal, redondeo clásico (>=0.5 aumenta parte entera)
     */
    imc: function( peso, estatura ) {

        return peso / Math.pow( estatura, 2 );
    },

    /**
     * <b>Nefrología</b><br/>
     * Calcula el índice de filtrado glomerular MDRD4<br>
     * ver {@link http://www.senefro.org/modules.php?name=calcfg}
     * 
     * @param {int} creatinina Valor de la creatinina según examen del paciente (en mg/dL)
     * @param {int} edad Edad del paciente (en años)
     * @param {string} sexo M: Masculino, F: femenino
     * @param {boolean} esRazaNegra true: si el paciente se dice que es de raza negra, false en otro caso
     * @return {float} Estimación de filtrado glomerular estimado, con todos los decimales
     */
    mdrd4: function( creatinina, edad, sexo, esRazaNegra ) { 

        var factorRaza = 1.212;

        creatinina = parseFloat( creatinina );
        if( isNaN( creatinina ) || creatinina <= 0 )
            throw new TypeError( "Parámetro creatinina no válido, ingrese un valor entero mayor a 0" );

        edad = parseInt( edad );
        if( isNaN( edad ) || edad <= 0 )
            throw new TypeError( "Parámetro edad no válido, ingrese un valor entero mayor a 0" );

        switch( sexo ) {
            case "M" : sexo = false; break;
            case "F" : sexo = true; break;
            default: throw new TypeError( "Parámetro sexo no válido. Valores válidos: M:masculino, F:Femenino" );
        }

        if( esRazaNegra !== true && esRazaNegra !== false )
            throw new TypeError( "Parámetro esRazaNegra no válido, ingrese true para paciente de raza negra, false en otros caso" );

        return 186 * ( Math.pow( creatinina, -1.154 ) ) * ( Math.pow( edad, -0.203 ) ) * ( ( sexo ? 0.742 : 1 ) ) * ( ( esRazaNegra ? factorRaza : 1 ) );
    },

    /**
     * <b>Nefrología</b><br/>
     * Calcula el índice de filtrado glomerular MDRD4-IDMS<br>
     * ver {@link http://www.senefro.org/modules.php?name=calcfg}
     * 
     * @param {int} creatinina Valor de la creatinina según examen del paciente (en mg/dL)
     * @param {int} edad Edad del paciente (en años)
     * @param {string} sexo M: Masculino, F: femenino
     * @param {boolean} esRazaNegra true: si el paciente se dice que es de raza negra, false en cualquier otro caso
     * @return {float} Estimación de filtrado glomerular estimado, con todos los decimales
     */
    mdrd4idms: function( creatinina, edad, sexo, esRazaNegra ) { 

        var factorRaza = 1.212;

        creatinina = parseFloat( creatinina );
        if( isNaN( creatinina ) || creatinina < 0 )
            throw new TypeError( "Parámetro creatinina no válido, ingrese un valor entero mayor a 0" );

        edad = parseInt( edad );
        if( isNaN( edad ) || edad < 0 )
            throw new TypeError( "Parámetro edad no válido, ingrese un valor entero mayor a 0" );

        switch( sexo ) {
            case "M" : sexo = false; break;
            case "F" : sexo = true; break;
            default: throw new TypeError( "Parámetro sexo no válido. Valores válidos: M:masculino, F:Femenino" );
        }

        if( esRazaNegra !== true && esRazaNegra !== false )
            throw new TypeError( "Parámetro esRazaNegra no válido, ingrese true para paciente de raza negra, false en cualquier caso" );

        return 175 * Math.pow( creatinina, -1.154 ) * Math.pow( edad, -0.203 ) * ( sexo ? 0.742 : 1 ) * ( esRazaNegra ? factorRaza : 1 );
    },

    /**
     * <b>Nefrología</b><br/>
     * Calcula el índice de filtrado glomerular MDRD6<br>
     * ver {@link http://www.senefro.org/modules.php?name=calcfg}
     * 
     * @param {int} creatinina Valor de la creatinina según examen del paciente (en mg/dL)
     * @param {int} edad Edad del paciente (en años)
     * @param {int} urea Valor del examen de laboratorio (en mg/dL)
     * @param {int} albumina Valor del examen de laboratorio (en g/dL)
     * @param {string} sexo M: Masculino, F: femenino
     * @param {boolean} esRazaNegra true: si el paciente se dice que es de raza negra, false en cualquier otro caso
     * @return {float} Estimación de filtrado glomerular estimado, con todos los decimales
     */
    mdrd6: function( creatinina, edad, urea, albumina, sexo, esRazaNegra ) { 

        var factorRaza = 1.180;

        creatinina = parseFloat( creatinina );
        if( isNaN( creatinina ) || creatinina <= 0 )
            throw new TypeError( "Parámetro creatinina no válido, ingrese un valor entero mayor a 0" );

        edad = parseInt( edad );
        if( isNaN( edad ) || edad <= 0 )
            throw new TypeError( "Parámetro edad no válido, ingrese un valor entero mayor a 0" );

        urea = parseInt( urea );
        if( isNaN( urea ) || urea <= 0 )
            throw new TypeError( "Parámetro urea no válido, ingrese un valor entero mayor a 0" );

        albumina = parseInt( albumina );
        if( isNaN( albumina ) || albumina <= 0 )
            throw new TypeError( "Parámetro albumina no válido, ingrese un valor entero mayor a 0" );

        switch( sexo ) {
            case "M" : sexo = false; break;
            case "F" : sexo = true; break;
            default: throw new TypeError( "Parámetro sexo no válido. Valores válidos: M:masculino, F:Femenino" );
        }

        if( esRazaNegra !== true && esRazaNegra !== false )
            throw new TypeError( "Parámetro esRazaNegra no válido, ingrese true para paciente de raza negra, false en cualquier caso" );

        return 170 * Math.pow( creatinina, -0.999 ) * Math.pow( edad, -0.176 ) * Math.pow( urea * 0.467, -0.170 ) * Math.pow( albumina, 0.318 ) * ( sexo ? 0.762 : 1 ) * ( esRazaNegra ? factorRaza : 1 );
    },

    /**
     * Realiza cálculo de superficie corporal. (método Dubois y Dubois)
     * {@link http://www.egfr.roche.es/calculadoras_medicas/superficie_corporal.htm}
     * 
     * @param altura en centimetros
     * @param peso en kilogramos
     * @return superficie corporal en metros cuadrados
     */
    superficie_corporal: function( altura, peso ) {

        altura = parseInt( altura );
        if( isNaN( altura ) || altura <= 0 )
            throw new TypeError( "Parámetro altura no válido, ingrese un valor entero mayor a 0" );

        peso = parseFloat( peso );
        if( isNaN( peso ) || peso <= 0 )
            throw new TypeError( "Parámetro peso no válido, ingrese un valor mayor a 0" );

        return ( 0.7184 * Math.pow( altura, 0.725 ) * Math.pow( peso, 0.425 ) )/100;
    },

    /**
     * <b>Nefrología</b><br/>
     * Cálculo Cockcroft-Gault
     * {@link http://www.senefro.org/modules.php?name=nefrocalc}
     * botón Nefrología, link Aclaramiento de creatinina estimado (Cockcroft-Gault)
     * <br/>
     * Clasificación en función de documento "Nefrología PIT final" envíado desde Colombia
     * 
     * @param altura en centímetros
     * @param peso en kilogramos
     * @param edad en años
     * @param creatinina en mg/dL
     * @param sexo M: Masculino, F: Femenino
     * 
     * @return indice de masa corporal (kgs/m2)
     */
    cockcroft_gault: function( altura, peso, edad, creatinina, sexo ) {

        altura = parseInt( altura );
        if( isNaN( altura ) || altura <= 0 )
            throw new TypeError( "Parámetro altura no válido, ingrese un valor entero mayor a 0" );

        peso = parseFloat( peso );
        if( isNaN( peso ) || peso <= 0 )
            throw new TypeError( "Parámetro peso no válido, ingrese un valor mayor a 0" );

        edad = parseInt( edad );
        if( isNaN( edad ) || edad <= 0 )
            throw new TypeError( "Parámetro edad no válido, ingrese un valor entero mayor a 0" );

        creatinina = parseFloat( creatinina );
        if( isNaN( creatinina ) || creatinina <= 0 )
            throw new TypeError( "Parámetro creatinina no válido, ingrese un valor entero mayor a 0" );
		/*
        switch( sexo ) {
            case "M" : sexo = false; break;
            case "F" : sexo = true; break;
            default: throw new TypeError( "Parámetro sexo no válido. Valores válidos: M:masculino, F:Femenino" );
        }
		*/
        // cm => mts.
        altura = altura/100;

        var calculo = NaN;
        if( edad >= 18 ) {
            if( sexo == "M" ) {

                calculo = ( ( 140-edad ) * peso ) / ( 72 * creatinina );
            } else {
                calculo = ( ( ( 140-edad ) * ( peso ) ) / ( 72 * creatinina ) ) * 0.85 ;
            }
        } else {
            calculo = ( 0.55 * altura ) / creatinina;
        }

        return calculo;
    },

    /**
     * <b>Nefrología</b><br/>
     * Cálculo Cockcroft-Gault corregido por SC
     * 
     * Obtenido desde {@link http://www.senefro.org/modules.php?name=calcfg}
     * //TODO Validar que sea así.
     * 
     * @param altura en centímetros
     * @param peso en kilogramos
     * @param edad en años
     * @param creatinina en mg/dL
     * @param sexo M: Masculino, F: Femenino
     * @param superficie_corporal
     * @returns cálculo Cockcroft-Gault por SC
     */
    cockcroft_gault_por_sc: function( altura, peso, edad, creatinina, sexo, superficie_corporal ) {

        altura = parseInt( altura );
        if( isNaN( altura ) || altura <= 0 )
            throw new TypeError( "Parámetro altura no válido, ingrese un valor entero mayor a 0" );

        peso = parseFloat( peso );
        if( isNaN( peso ) || peso <= 0 )
            throw new TypeError( "Parámetro peso no válido, ingrese un valor mayor a 0" );

        edad = parseInt( edad );
        if( isNaN( edad ) || edad <= 0 )
            throw new TypeError( "Parámetro edad no válido, ingrese un valor entero mayor a 0" );

        creatinina = parseFloat( creatinina );
        if( isNaN( creatinina ) || creatinina <= 0 )
            throw new TypeError( "Parámetro creatinina no válido, ingrese un valor entero mayor a 0" );

        switch( sexo ) {
            // Validaciones necesarias (m o f, no usar true o false), acepta solo M o F, no otro parámetro
            case "M" : sexo = "M"; break;
            case "F" : sexo = "F"; break;
            default: throw new TypeError( "Parámetro sexo no válido. Valores válidos: M:masculino, F:Femenino" );
        }

        superficie_corporal = parseFloat( superficie_corporal );
        if( isNaN( superficie_corporal ) || superficie_corporal <= 0 )
            throw new TypeError( "Parámetro superficie corporal no válido, ingrese un valor entero mayor a 0" );



        var cg = this.cockcroft_gault( altura, peso, edad, creatinina, sexo );
        return ( cg * 1.73 ) / superficie_corporal;
    },

    /**
     * <b>Nefrología</b><br/>
     * Ecuación de estimación del filtrado glomerular <br/>
     * 
     * @param {boolean} esRazaNegra true: si el paciente se dice que es de raza negra, false en cualquier otro caso
     * @param {string} sexo M: Masculino, F: femenino
     * @param {Integer} creatinina Valor de la creatinina según examen del paciente (en mg/dL)
     * @param {Integer} edad Edad del paciente (en años)
     * @returns {Integer} Valor estimado de creatinina, en ml/min/1.73m<sup>2</sup>
     */
    ckdepi: function( esRazaNegra, sexo, creatinina, edad ) {

        if( !( sexo === "M" || sexo === "F" ) )
            throw new TypeError( "Parámetro sexo no válido. Valores válidos: M:masculino, F:Femenino" );

        if( esRazaNegra !== true && esRazaNegra !== false )
            throw new TypeError( "Parámetro esRazaNegra no válido, ingrese true para paciente de raza negra, false en cualquier caso" );

        creatinina = parseFloat( creatinina );
        if( isNaN( creatinina ) || creatinina < 0 )
            throw new TypeError( "Parámetro creatinina no válido, ingrese un valor entero mayor a 0" );

        edad = parseInt( edad );
        if( isNaN( edad ) || edad < 0 )
            throw new TypeError( "Parámetro edad no válido, ingrese un valor entero mayor a 0" );

        valor = NaN;
        if( esRazaNegra ) {
            switch( sexo ) {
                // M: Masculino
                case "M" :
                    if( creatinina <= 0.9 )
                        valor = 163 * Math.pow( creatinina/0.9, -0.411 ) * Math.pow( 0.993, edad );
                    else
                        valor = 163 * Math.pow( creatinina/0.9, -1.209 ) * Math.pow( 0.993, edad );
                    break;
                // F: Femenino
                case "F" :
                    if( creatinina <= 0.7 )
                        valor = 166 * Math.pow( creatinina/0.7, -0.329 ) * Math.pow( 0.993, edad );
                    else
                        valor = 166 * Math.pow( creatinina/0.7, -1.209 ) * Math.pow( 0.993, edad );
                    break;
            }
		  valor = valor + 0.25;//+ 0.25 Requerimiento solicitado por Carolina Parra con el objetivo de igualar resultados a los entregados por sitio web SEN: http://www.senefro.org/modules.php?name=nefrocalc
			
        } else {
            switch( sexo ) {
                // M: Masculino
                case "M" :
                    if( creatinina <= 0.9 )
                        valor = 141 * Math.pow( creatinina/0.9, -0.411 ) * Math.pow( 0.993, edad );
                    else
                        valor = 141 * Math.pow( creatinina/0.9, -1.209 ) * Math.pow( 0.993, edad );
                    break;
                // F: Femenino
                case "F" :
                    if( creatinina <= 0.7 )
                        valor = 144 * Math.pow( creatinina/0.7, -0.329 ) * Math.pow( 0.993, edad );
                    else
                        valor = 144 * Math.pow( creatinina/0.7, -1.209 ) * Math.pow( 0.993, edad );
					
					valor = valor - 0.25;//- 0.25 Requerimiento solicitado por Carolina Parra con el objetivo de igualar resultados a los entregados por sitio web SEN: http://www.senefro.org/modules.php?name=nefrocalc
                    break;
            }
        }

        return valor;
    },

    /**
     * Retorna un número aleatorio de largo <code>largo</code>
     * 
     * @param int largo del string a retornar
     * @returns int número aleatorio generado
     */
    generar_numero_aleatorio: function( largo ) {

        a = Math.random().toString();
        if( a.length-2 < largo ) return false;

        return a.slice( -(largo) );
    }
}
